#!/usr/bin/perl
#
# Passively fingerprints OSes based on signatures
# passfingerprint.pl version 0.2
#
# Modifications:
#      - Disabled TOS as a signature, this is not OS dependent.
#
# Craig Smith (April 2000) - Licenced under the GPL (see COPYING)
#
#
#
# Usage:
#
#   passivefingerprint [store|print]
#
#   w/o any arguments it will just print packets to the screen with an OS guess
#   store - save to a DB file (traffic.db) all IPs and OSes
#           stores both known and unknown OSes
#   print - prints the DB file (traffic.db)
#

### Required modules
use Net::RawIP;

### Variables
$psize = 2048;
$device = "eth0";	# System interface, be sure you set this
$timeout = 500;
$filter = "tcp";
$detail = 1;		# verbose level
$diff = 30;		# TTL allowable difference (hops)
$line;			# Temp scalar for reading files
@printdata;		# Holds hold fingerprint info into an array
%fingerprint;		# holds are fingerprint info from file (used in array)

### Command line options
$usedb=0;
if ($ARGV[0] eq "print") {
   &printdb;
   exit;
} elsif ($ARGV[0] eq "store") {
   use DB_File;
   $usedb=1;
}

### Open signature database .dat file
open FPDATA, "./fingerprints.dat" || die "Couldn't open fingerprints : $!";
while(<FPDATA>) {
  $line = $_;
  if($line=~/^#/) { next; }
  if($line) {
	if($line=~/(\S*)(\s*)(\S*)(\s*)(\S*)(\s*)(\S*)(\s*)(\S*)(\s*)(\S*)(\s*)(\S*)/) {
	  $fingerprint = {};
	  $fingerprint->{OS} = $1;
	  $fingerprint->{VERSION} = $3;
	  $fingerprint->{PLATFORM} = $5;
	  $fingerprint->{TTL} = $7;
	  $fingerprint->{WINDOW} = $9;
	  $fingerprint->{DF} = $11;
	  $fingerprint->{TOS} = $13;
	  # WINDOW is special, we need to check for ranges
	  if($fingerprint->{WINDOW}=~/(\d*)-(\d*)/) {
	    $fingerprint->{WINDOW_MIN}=$1;
	    $fingerprint->{WINDOW_MAX}=$2;
	  } else {	# Exact number
	    $fingerprint->{WINDOW_MIN}=$fingerprint->{WINDOW};
	    $fingerprint->{WINDOW_MAX}=$fingerprint->{WINDOW};
	  }
	  push @printdata, $fingerprint;
	} else {	# Invalid line
	  die "Invalid line in fingerprint file";
	}
	if($detail) {	# Print verbose stuff
	 print "$1,$3,$5,$7,$fingerprint->{WINDOW_MIN}-",
		"$fingerprint->{WINDOW_MAX},$11,$13\n"; 
	}
  }
}
close FPDATA;


### Actual program
my $pkt = new Net::RawIP({ip=>{},tcp=>{}});
my $pcap = $pkt->pcapinit($device,$filter, $psize, $timeout);
my $offset = Net::RawIP::linkoffset($pcap);
print "Link offset: $offset\n";
loop $pcap,-1,\&fingerprint, \@pkt;

### Routines
sub fingerprint {
  $pckt = $_[2];
  $pkt->bset(substr($pckt,$offset));
  my ($tos, $ttl, $saddr, $daddr, $sport, $dport, $windowsize) = 
	    $pkt->get(
		{ip => [qw(tos ttl saddr daddr)],
		 tcp => [qw(source dest window)]
		 });
  my @psrc = unpack("C4",pack("N", $saddr));
  my @pdest = unpack("C4",pack("N", $daddr));
  $saddr = join('.',@psrc);
  $daddr = join('.',@pdest);
  $df=unpack("C",substr($pckt,$offset+6,1));
  $df= ($df == 64 ? "y" : "n");	# There is a better way
  my $guess = "Unknown OS";

  if($windowsize) {		# Packet probably is useless to use...skip it
   foreach (@printdata) {
    if(
	($ttl <= $_->{TTL} && $ttl >= ($_->{TTL}-$diff)) &&
	($windowsize >= $_->{WINDOW_MIN} && $windowsize <= $_->{WINDOW_MAX}) &&
        # TOS has been disabled, as that is application specific
        # ($tos == $_->{TOS}) &&
	($df eq $_->{DF})) {
	  $guess = "$_->{OS} $_->{VERSION} [$_->{PLATFORM}]";
	}
   }
   if($usedb) {
	  my %db;
	  tie %db, "DB_File", "./traffic.db";
	  $db{"$saddr"}="$guess";
	  $db{"$saddr"} .= " - TOS($tos) TTL($ttl) DF($df) WINDOW($windowsize)" if $guess =~ /Unknown/;
	  untie %db;
   }
   print "PACKET $saddr:$sport -> $daddr:$dport SRC($guess)\n";
   print "\tTOS($tos) TTL($ttl) DF($df) WINDOW($windowsize)\n" if $detail == 1 || $guess=~/Unknown/;;
  } # EndIf $windowsize
}

sub printdb {
  use DB_File;
  my %db;
  print "Contents of ./traffic.db\n";
  print "------------------------\n";
  tie %db, "DB_File", "./traffic.db";
  while (($ip, $os)=each %db) {
	print "$ip == $os\n";
  }
  untie %db;
}

